/*
** Copyright (C) 2001-2025 Zabbix SIA
**
** This program is free software: you can redistribute it and/or modify it under the terms of
** the GNU Affero General Public License as published by the Free Software Foundation, version 3.
**
** This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
** without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
** See the GNU Affero General Public License for more details.
**
** You should have received a copy of the GNU Affero General Public License along with this program.
** If not, see <https://www.gnu.org/licenses/>.
**/

package plugin

import (
	"context"
	"errors"

	"github.com/jackc/pgx/v4"
	"golang.zabbix.com/sdk/zbxerr"
)

// bgwriterHandler executes select  with statistics from pg_stat_bgwriter
// and returns JSON if all is OK or nil otherwise.
func bgwriterHandler(ctx context.Context, conn PostgresClient,
	_ string, _ map[string]string, _ ...string) (interface{}, error) {
	var bgwriterJSON string

	query := `
  SELECT row_to_json (T)
    FROM (
          SELECT
              checkpoints_timed
            , checkpoints_req
            , checkpoint_write_time
            , checkpoint_sync_time
            , buffers_checkpoint
            , buffers_clean
            , maxwritten_clean
            , buffers_backend
            , buffers_backend_fsync
            , buffers_alloc
          FROM pg_catalog.pg_stat_bgwriter
		  ) T ;`

	row, err := conn.QueryRow(ctx, query)
	if err != nil {
		return nil, zbxerr.ErrorCannotFetchData.Wrap(err)
	}

	err = row.Scan(&bgwriterJSON)
	if err != nil {
		if errors.Is(err, pgx.ErrNoRows) {
			return nil, zbxerr.ErrorEmptyResult.Wrap(err)
		}

		return nil, zbxerr.ErrorCannotFetchData.Wrap(err)
	}

	return bgwriterJSON, nil
}
