from arcnagios.ldaputils import LDAPObject
import logging
from inspect import isclass
from utils import lazy_staticmethod

log = logging.getLogger(__name__)

MU_1 = 0
MU_01 = 1
MU_SOME = 2
MU_ANY = 3
MU_MINUS = 4

# TODO. Change to MU_SOME when the infosys is ready for it.
MU_SOME_WIP = MU_ANY

def multiplicity_indicator(m):
    return {MU_1: '1', MU_01: '0,1', MU_ANY: '*', MU_SOME: '+',
	    MU_MINUS: '-'}[m]

def matching_multiplicity(mult, n):
    if mult == MU_1:
	return n == 1
    if mult == MU_01:
	return n in [0, 1]
    if mult == MU_SOME:
	return n > 0
    if mult == MU_ANY:
	return True

# Relation Kinds
ASSOCIATION = 1
AGGREGATION = 2
COMPOSITION = 3

# TODO: The specification of relation of the GLUE2ForeignKey objects may be
# incomplete, unspecified ones default to ASSOCIATION.

class GLUE2ForeignKey(object):
    def __init__(self, name, other_class, local_mult, other_mult,
		 bidirectional = False, dependent_fk_attribute = False,
		 relation = ASSOCIATION):
	self.name = name
	self.other_class = other_class
	self.local_mult = local_mult
	self.other_mult = other_mult
	self.bidirectional = bidirectional
	self.dependent_fk_attribute = dependent_fk_attribute
	self.relation = relation

_fk = GLUE2ForeignKey

# Auxiliary classes to be treated as structual.
glue2_exclusive_auxiliary_objectclasses = {
	'GLUE2AdminDomain': 'GLUE2Domain',
	'GLUE2UserDomain': 'GLUE2Domain',
	'GLUE2AccessPolicy': 'GLUE2Policy',
	'GLUE2MappingPolicy': 'GLUE2Policy',
	'GLUE2ComputingService': 'GLUE2Service',
	'GLUE2ComputingEndpoint': 'GLUE2Endpoint',
	'GLUE2ComputingShare': 'GLUE2Share',
	'GLUE2ComputingManager': 'GLUE2Manager',
	'GLUE2ExecutionEnvironment': 'GLUE2Resource',
	'GLUE2ComputingActivity': 'GLUE2Activity',
	'GLUE2StorageService': 'GLUE2Service',
	'GLUE2StorageEndpoint': 'GLUE2Endpoint',
	'GLUE2StorageShare': 'GLUE2Share',
	'GLUE2StorageManager': 'GLUE2Manager',
	'GLUE2DataStore': 'GLUE2Resource',
}

class GLUE2Entity(LDAPObject):
    structural_objectclass = 'GLUE2Entity'
    glue2_exclusive_auxiliary_objectclass = None
    glue2_really_abstract = False
    glue2_primary_key = None
    glue2_foreign_keys = lazy_staticmethod(lambda: [])

    @classmethod
    def glue2_class_name(cls):
	return cls.glue2_exclusive_auxiliary_objectclass \
	    or cls.structural_objectclass

    @classmethod
    def glue2_check_class(cls):
	error_count = 0
	log.info('+ Checking class %s.'%cls.__name__)
	for fk in cls.glue2_foreign_keys():
	    log.info('++ Checking FK %s.'%fk.name)
	    if not fk.dependent_fk_attribute:
		if not isinstance(fk.other_class.glue2_primary_key, str):
		    log.error('Missing primary key for %s'%fk.name)
		    error_count += 1
	    if fk.bidirectional:
		found = False
		for rfk in fk.other_class.glue2_foreign_keys():
		    if issubclass(cls, rfk.other_class):
			found = True
		if not found:
		    log.error('Did not find reverse link for %s'%fk.name)
		    error_count += 1
	if cls.glue2_class_name() != cls.__name__:
	    log.error('Mismatched class name %s vs %s.'
		      % (cls.glue2_class_name(), cls.__name__))
	    error_count += 1
	assert error_count == 0

    @classmethod
    def glue2_all_foreign_keys(cls):
	for fk in cls.glue2_foreign_keys():
	    yield fk
	for base in cls.__bases__:
	    if issubclass(base, GLUE2Entity):
		for fk in base.glue2_all_foreign_keys():
		    yield fk

    def glue2_get_fk_links(self, fk):
	v = getattr(self, fk.name)
	if isinstance(v, list):
	    return v
	elif v is None:
	    return []
	else:
	    return [v]

class GLUE2Group(GLUE2Entity):
    structural_objectclass = 'GLUE2Group'
    glue2_primary_key = 'GLUE2GroupID'
    glue2_foreign_keys = lazy_staticmethod(lambda: [])

class GLUE2Extension(GLUE2Entity):
    structural_objectclass = 'GLUE2Extension'
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2ExtensionEntityForeignKey', GLUE2Entity, MU_ANY, MU_1,
	    dependent_fk_attribute = True, relation = COMPOSITION),
    ])

class GLUE2Location(GLUE2Entity):
    structural_objectclass = 'GLUE2Location'
    glue2_primary_key = 'GLUE2LocationID'
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2LocationServiceForeignKey', GLUE2Service, MU_01, MU_ANY),
	_fk('GLUE2LocationDomainForeignKey', GLUE2Domain, MU_01, MU_ANY),
    ])

class GLUE2Contact(GLUE2Entity):
    structural_objectclass = 'GLUE2Contact'
    glue2_primary_key = 'GLUE2ContactID'
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2ContactServiceForeignKey', GLUE2Service, MU_ANY, MU_ANY),
	_fk('GLUE2ContactDomainForeignKey', GLUE2Domain, MU_ANY, MU_ANY),
    ])

class GLUE2Domain(GLUE2Entity):
    structural_objectclass = 'GLUE2Domain'
    glue2_primary_key = 'GLUE2DomainID'
    glue2_really_abstract = True
    glue2_foreign_keys = lazy_staticmethod(lambda: [])

class GLUE2AdminDomain(GLUE2Domain):
    glue2_exclusive_auxiliary_objectclass = 'GLUE2AdminDomain'
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2AdminDomainAdminDomainForeignKey', GLUE2AdminDomain,
	    MU_ANY, MU_01, relation = AGGREGATION),
    ])

class GLUE2UserDomain(GLUE2Domain):
    glue2_exclusive_auxiliary_objectclass = 'GLUE2UserDomain'
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2UserDomainUserDomainForeignKey', GLUE2UserDomain,
	    MU_ANY, MU_01, relation = AGGREGATION),
    ])

class GLUE2Service(GLUE2Entity):
    structural_objectclass = 'GLUE2Service'
    glue2_primary_key = 'GLUE2ServiceID'
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2ServiceAdminDomainForeignKey', GLUE2AdminDomain,
	    MU_ANY, MU_1, relation = AGGREGATION),
	_fk('GLUE2ServiceServiceForeignKey', GLUE2Service, MU_ANY, MU_ANY,
	    bidirectional = True),
    ])

class GLUE2Endpoint(GLUE2Entity):
    structural_objectclass = 'GLUE2Endpoint'
    glue2_primary_key = 'GLUE2EndpointID'
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2EndpointServiceForeignKey', GLUE2Service, MU_ANY, MU_1,
	    relation = AGGREGATION),
    ])

class GLUE2Share(GLUE2Entity):
    structural_objectclass = 'GLUE2Share'
    glue2_primary_key = 'GLUE2ShareID'
    glue2_really_abstract = True
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2ShareServiceForeignKey', GLUE2Service, MU_ANY, MU_1,
	    relation = AGGREGATION),
	_fk('GLUE2ShareEndpointForeignKey', GLUE2Share, MU_ANY, MU_ANY),
	_fk('GLUE2ShareResourceForeignKey', GLUE2Resource, MU_ANY, MU_ANY),
    ])

class GLUE2Manager(GLUE2Entity):
    structural_objectclass = 'GLUE2Manager'
    glue2_primary_key = 'GLUE2ManagerID'
    glue2_really_abstract = True
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2ManagerServiceForeignKey', GLUE2Service, MU_ANY, MU_1,
	    relation = AGGREGATION),
    ])

class GLUE2Resource(GLUE2Entity):
    structural_objectclass = 'GLUE2Resource'
    glue2_primary_key = 'GLUE2ResourceID'
    glue2_really_abstract = True
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2ResourceManagerForeignKey', GLUE2Manager, MU_ANY, MU_1,
	    relation = COMPOSITION),
    ])

class GLUE2Activity(GLUE2Entity):
    structural_objectclass = 'GLUE2Activity'
    glue2_primary_key = 'GLUE2ActivityID'
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2ActivityUserDomainForeignKey', GLUE2UserDomain,
	    MU_ANY, MU_01),
	_fk('GLUE2ActivityEndpointForeignKey', GLUE2Endpoint, MU_ANY, MU_01),
	_fk('GLUE2ActivityShareForeignKey', GLUE2Share, MU_ANY, MU_01),
	_fk('GLUE2ActivityResourceForeignKey', GLUE2Resource, MU_ANY, MU_01),
	_fk('GLUE2ActivityActivityForeignKey', GLUE2Activity, MU_ANY, MU_ANY,
	    bidirectional = True)
    ])

class GLUE2Policy(GLUE2Entity):
    structural_objectclass = 'GLUE2Policy'
    glue2_primary_key = 'GLUE2PolicyID'
    glue2_really_abstract = True
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2PolicyUserDomainForeignKey', GLUE2UserDomain,
	    MU_ANY, MU_SOME_WIP),
    ])

class GLUE2AccessPolicy(GLUE2Policy):
    glue2_exclusive_auxiliary_objectclass = 'GLUE2AccessPolicy'
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2AccessPolicyEndpointForeignKey', GLUE2Endpoint, MU_ANY, MU_1),
    ])

class GLUE2MappingPolicy(GLUE2Policy):
    glue2_exclusive_auxiliary_objectclass = 'GLUE2MappingPolicy'
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2MappingPolicyShareForeignKey', GLUE2Share, MU_ANY, MU_1),
    ])


# Computing Service
# =================

class GLUE2ComputingService(GLUE2Service):
    glue2_exclusive_auxiliary_objectclass = 'GLUE2ComputingService'
    glue2_foreign_keys = lazy_staticmethod(lambda: [])
class GLUE2ComputingEndpoint(GLUE2Endpoint):
    glue2_exclusive_auxiliary_objectclass = 'GLUE2ComputingEndpoint'
    glue2_foreign_keys = lazy_staticmethod(lambda: [])
class GLUE2ComputingShare(GLUE2Share):
    glue2_exclusive_auxiliary_objectclass = 'GLUE2ComputingShare'
    glue2_foreign_keys = lazy_staticmethod(lambda: [])
class GLUE2ComputingManager(GLUE2Manager):
    glue2_exclusive_auxiliary_objectclass = 'GLUE2ComputingManager'
    glue2_foreign_keys = lazy_staticmethod(lambda: [])
class GLUE2Benchmark(GLUE2Entity):
    structural_objectclass = 'GLUE2Benchmark'
    glue2_primary_key = 'GLUE2BenchmarkID'
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2BenchmarkComputingManagerForeignKey',
	    GLUE2ComputingManager, MU_ANY, MU_01),
	_fk('GLUE2BenchmarkExecutionEnvironmentForeignKey',
	    GLUE2ExecutionEnvironment, MU_ANY, MU_01),
    ])
class GLUE2ExecutionEnvironment(GLUE2Resource):
    glue2_exclusive_auxiliary_objectclass = 'GLUE2ExecutionEnvironment'
    glue2_foreign_keys = lazy_staticmethod(lambda: [])
class GLUE2ApplicationEnvironment(GLUE2Entity):
    structural_objectclass = 'GLUE2ApplicationEnvironment'
    glue2_primary_key = 'GLUE2ApplicationEnvironmentID'
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2ApplicationEnvironmentComputingManagerForeignKey',
	    GLUE2ComputingManager, MU_ANY, MU_1, relation = COMPOSITION),
	_fk('GLUE2ApplicationEnvironmentExecutionEnvironmentForeignKey',
	    GLUE2ExecutionEnvironment, MU_ANY, MU_ANY),
    ])
class GLUE2ApplicationHandle(GLUE2Entity):
    structural_objectclass = 'GLUE2ApplicationHandle'
    glue2_primary_key = 'GLUE2ApplicationHandleID'
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2ApplicationHandleApplicationEnvironmentForeignKey',
	    GLUE2ApplicationEnvironment, MU_ANY, MU_1, relation = COMPOSITION),
    ])
class GLUE2ComputingActivity(GLUE2Activity):
    glue2_exclusive_auxiliary_objectclass = 'GLUE2ComputingActivity'
    glue2_foreign_keys = lazy_staticmethod(lambda: [])
class GLUE2ToStorageService(GLUE2Entity):
    structural_objectclass = 'GLUE2ToStorageService'
    glue2_primary_key = 'GLUE2ToStorageServiceID'
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2ToStorageServiceComputingServiceForeignKey',
	    GLUE2ComputingService, MU_ANY, MU_1),
	_fk('GLUE2ToStorageServiceStorageServiceForeignKey',
	    GLUE2StorageService, MU_MINUS, MU_1),
    ])
glue2_computing_classes = [
	GLUE2ComputingService, GLUE2ComputingEndpoint, GLUE2ComputingShare,
	GLUE2ComputingManager, GLUE2Benchmark, GLUE2ExecutionEnvironment,
	GLUE2ApplicationEnvironment, GLUE2ApplicationHandle,
	GLUE2ComputingActivity, GLUE2ToStorageService
]

# Storage Service
# ===============

class GLUE2StorageService(GLUE2Service):
    glue2_exclusive_auxiliary_objectclass = 'GLUE2StorageService'
    glue2_foreign_keys = lazy_staticmethod(lambda: [])
class GLUE2StorageServiceCapacity(GLUE2Entity):
    structural_objectclass = 'GLUE2StorageServiceCapacity'
    glue2_primary_key = 'GLUE2StorageServiceCapacityID'
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2StorageServiceCapacityStorageServiceForeignKey',
	    GLUE2StorageService, MU_ANY, MU_1),
    ])
class GLUE2StorageAccessProtocol(GLUE2Entity):
    structural_objectclass = 'GLUE2StorageAccessProtocol'
    glue2_primary_key = 'GLUE2StorageAccessProtocolID'
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2StorageAccessProtocolStorageServiceForeignKey',
	    GLUE2StorageService, MU_ANY, MU_1, relation = AGGREGATION),
    ])
class GLUE2StorageEndpoint(GLUE2Endpoint):
    glue2_exclusive_auxiliary_objectclass = 'GLUE2StorageEndpoint'
    glue2_foreign_keys = lazy_staticmethod(lambda: [])
class GLUE2StorageShare(GLUE2Share):
    glue2_exclusive_auxiliary_objectclass = 'GLUE2StorageShare'
    glue2_foreign_keys = lazy_staticmethod(lambda: [])
class GLUE2StorageShareCapacity(GLUE2Entity):
    structural_objectclass = 'GLUE2StorageShareCapacity'
    glue2_primary_key = 'GLUE2StorageShareCapacityID'
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2StorageShareCapacityStorageShareForeignKey',
	    GLUE2StorageShare, MU_ANY, MU_1, relation = AGGREGATION),
    ])
class GLUE2StorageManager(GLUE2Manager):
    glue2_exclusive_auxiliary_objectclass = 'GLUE2StorageManager'
    glue2_foreign_keys = lazy_staticmethod(lambda: [])
class GLUE2DataStore(GLUE2Resource):
    glue2_exclusive_auxiliary_objectclass = 'GLUE2DataStore'
    glue2_foreign_keys = lazy_staticmethod(lambda: [])
class GLUE2ToComputingService(GLUE2Entity):
    structural_objectclass = 'GLUE2ToComputingService'
    glue2_primary_key = 'GLUE2ToComputingServiceID'
    glue2_foreign_keys = lazy_staticmethod(lambda: [
	_fk('GLUE2ToComputingServiceStorageAccessProtocolForeignKey',
	    GLUE2StorageAccessProtocol, MU_ANY, MU_ANY),
	_fk('GLUE2ToComputingServiceComputingServiceForeignKey',
	    GLUE2ComputingService, MU_MINUS, MU_1),
	_fk('GLUE2ToComputingServiceStorageServiceForeignKey',
	    GLUE2StorageService, MU_MINUS, MU_1),
    ])
glue2_storage_classes = [
	GLUE2StorageService, GLUE2StorageServiceCapacity,
	GLUE2StorageAccessProtocol, GLUE2StorageEndpoint,
	GLUE2StorageShare, GLUE2StorageShareCapacity, GLUE2StorageManager,
	GLUE2DataStore, GLUE2ToComputingService
]

# Costructor Dispatcher
# =====================

glue2_entity_classes = \
	filter(lambda c: isclass(c) and issubclass(c, GLUE2Entity),
	       globals().itervalues())

glue2_entity_class_map = \
	dict((c.glue2_class_name(), c) for c in glue2_entity_classes)

def construct_from_ldap_entry(subschema, dn, ent):
    ocs = set(ent['objectClass'])
    exauxes = ocs.intersection(glue2_exclusive_auxiliary_objectclasses)
    if len(exauxes) > 1:
	raise ValueError('Mixing exclusive auxiliary classes %s.'
			 % ', '.join(exauxes))
    if len(exauxes) == 1:
	class_name = exauxes.pop()
	base_class_name = glue2_exclusive_auxiliary_objectclasses[class_name]
	if not base_class_name in ocs:
	    raise ValueError(
		    '%s should be considered a structural subclass of %s.'
		    %(class_name, base_class_name))
    else:
	class_name = ent['structuralObjectClass'][0]
    c = glue2_entity_class_map.get(class_name)
    if c:
	return c(subschema, dn, ent)


# Validation of Classes
# =====================

# To validate the classes and dump a diagram to tmp.glue2.*, use
#   python -c 'from arcnagios.glue2 import check_classes; check_classes()'
def check_classes():
    def nodename(s):
	s = s.startswith('GLUE2') and s[5:] or s
	return s
    def fklabel(cn, s):
	if not s.endswith('ForeignKey'):
	    log.error('%s should end with "ForeignKey"')
	    return s
	if not s.startswith(cn):
	    log.error('%s should start with %s'%(s, cn))
	    return '"%s"'%nodename(s)
	s = s[:-10] + 'FK'
	return '"%s-\\n%s"'%(nodename(cn), s[len(cn):])
    logging.basicConfig(loglevel = logging.INFO)
    dot_out = open('tmp.glue2.dot', 'w')
    dot_out.write('digraph {\n'
		  '  rankdir = "BT";\n'
		  '  edge [arrowsize=1.2, fontsize=13, labelfontsize=15];\n')
    if False:
	dot_out.write('  subgraph cluster_computing {%s;}\n'
	    % '; '.join(c.glue2_class_name() for c in glue2_computing_classes))
	dot_out.write('  subgraph cluster_storage {%s;}\n'
	    % '; '.join(c.glue2_class_name() for c in glue2_storage_classes))
    for c in glue2_entity_classes:
	c.glue2_check_class()
	if not c is GLUE2Entity:
	    for cp in c.__subclasses__():
		dot_out.write('  %s -> %s;\n'
			      %(nodename(cp.glue2_class_name()),
				nodename(c.glue2_class_name())))
	for fk in c.glue2_foreign_keys():
	    cp = fk.other_class
	    dot_out.write('  %s -> %s [arrowhead=odiamond, label=%s, '
			  'taillabel="%s", headlabel="%s"];\n'
			  %(nodename(c.glue2_class_name()),
			    nodename(cp.glue2_class_name()),
			    fklabel(c.glue2_class_name(), fk.name),
			    multiplicity_indicator(fk.local_mult),
			    multiplicity_indicator(fk.other_mult)))
    dot_out.write('}\n')
    dot_out.close()
    import os
    return os.system('dot -T svg -o tmp.glue2.svg tmp.glue2.dot')
